﻿using Cyotek.Windows.Forms;
using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace NilaOCR
{
    public partial class OCRForm : Form
    {
        public List<GutschriftDocumentModel> gutschriftDocList = null;
        public List<GutschriftDocumentModel> allDocList = null;
        private int completedDocCount = 0;
        private int pendingDocCount = 0;
        
        public OCRForm()
        {
            InitializeComponent();
            btnStop.Enabled = false;
            OCRSettings.LoadSettings();

            if (!Directory.Exists(OCRSettings.settings.OutputDocPath))
            {
                Directory.CreateDirectory(OCRSettings.settings.OutputDocPath);
            }
            ResetProgressBar();
            txtInvoice.Text = "";
            txtOrder.Text = "";
            txtDate.Text = "";
            txtCostCenter.Text = "";
            txtSAPNumber.Text = "";
            txtSumNetto.Text = "";
            btnUpdate.Enabled = false;
        }
        private void UpdateUI()
        {
            //positionToolStripStatusLabel.Text = pdfImageBox.AutoScrollPosition.ToString();
            //imageSizeToolStripStatusLabel.Text = pdfImageBox.GetImageViewPort().ToString();
            //zoomToolStripStatusLabel.Text = string.Format("{0}%", pdfImageBox.Zoom);

            firstPageToolStripButton.Enabled = pdfImageBox.CanMoveFirst;
            previousPageToolStripButton.Enabled = pdfImageBox.CanMovePrevious;
            nextPageToolStripButton.Enabled = pdfImageBox.CanMoveNext;
            lastPageToolStripButton.Enabled = pdfImageBox.CanMoveLast;
            pageToolStripTextBox.Text = pdfImageBox.CurrentPage.ToString();

            sizeToFitToolStripButton.Checked = pdfImageBox.SizeToFit;
            actualSizeToolStripButton.Checked = (pdfImageBox.Zoom == 100);
        }
        private void btnStart_Click(object sender, EventArgs e)
        {

            btnStop.Enabled = true;
            btnStart.Enabled = false;

            lstDocFiles.Items.Clear();

            completedDocCount = 0;
            pendingDocCount = 0;

            SetStatus("Processing OCR.. Please Wait..");
            string[] pdfFfiles = Directory.GetFiles(OCRSettings.settings.GutschriftDocPath, "*.pdf");

            prgBarOcr.Maximum = pdfFfiles.Length;
            prgBarOcr.Visible = true;
            prgBarOcr.Update();

            BackgroundWorker myWorker = new BackgroundWorker();

            myWorker.DoWork += myWorker_DoWork;
            myWorker.RunWorkerCompleted += myWorker_RunWorkerCompleted;
            myWorker.ProgressChanged += myWorker_ProgressChanged;
            myWorker.WorkerReportsProgress = true;
            myWorker.RunWorkerAsync();

        }

        void myWorker_ProgressChanged(object sender, ProgressChangedEventArgs e)
        {
            UpdateProgressBar();
        }

        void myWorker_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            ResetProgressBar();

            UpdateDocList();

            SetStatus("OCR Process completed");

            btnStop.Enabled = false;
            btnStart.Enabled = true;
        }

        void myWorker_DoWork(object sender, DoWorkEventArgs e)
        {
            BackgroundWorker sendingWorker = (BackgroundWorker)sender;//Capture the BackgroundWorker that fired the event
            ProcessOCR(sendingWorker);
        }

        public void UpdateDocList()
        {
            if (gutschriftDocList.Count == 0)
            {
                MessageBox.Show("Alle Dokumente verarbeitet. Keine Auffälligkeiten gefunden.");
                this.Close();
                ShowCompletedRecords();

            }
            lstDocFiles.Items.Clear();
            for (int i = 0; i < gutschriftDocList.Count; i++)
            {
                if (gutschriftDocList[i].OrderNumber != null && !gutschriftDocList[i].OrderNumber.Equals(""))
                    lstDocFiles.Items.Add(gutschriftDocList[i].OrderNumber);
                else
                    lstDocFiles.Items.Add(i.ToString());
            }
            txtOrder.Text = "";
            txtInvoice.Text = "";
            txtDate.Text ="";
            txtCostCenter.Text = "";
            txtSAPNumber.Text = "";
            txtSumNetto.Text = "";
            btnUpdate.Enabled = false;
        }

        private void ProcessOCR(BackgroundWorker sendingWorker)
        {
            gutschriftDocList = new List<GutschriftDocumentModel>();
            allDocList = new List<GutschriftDocumentModel>();
            string[] pdfFfiles = Directory.GetFiles(OCRSettings.settings.GutschriftDocPath, "*.pdf");

            if (pdfFfiles.Length == 0)
            {
                MessageBox.Show("Keine Daten zur Verarbeitung verfügbar");
                return;
            }


            AccessDbConnector adCon = new AccessDbConnector();
            adCon.ConnectToDatabase();
            ArrayList invDocs = new ArrayList();

            for (int i = 0; i < pdfFfiles.Length; i++)
            {
                sendingWorker.ReportProgress(i);
                GutschriftDocumentModel gDoc = new GutschriftDocumentModel();

                try
                {
                    gDoc.Process(pdfFfiles[i]);

                    if (!gDoc.ValidDocument)
                    {

                        gDoc.InitInvNumber = adCon.InsertInvoiceTempRecord(gDoc.GetInvoiceTempData()).ToString();
                        gutschriftDocList.Add(gDoc);
                        pendingDocCount--;

                    }
                    else
                    {
                        int id = adCon.InsertInvoiceRecord(gDoc.GetInvoiceData());
                        if (id > 0)
                        {
                            gDoc.ID = id;
                            MoveFilesToUpdated(gDoc);
                        }
                        allDocList.Add(gDoc);
                        completedDocCount++;
                    }
                }
                catch(Exception ex)
                {
                    invDocs.Add(pdfFfiles[i]);
                    //MessageBox.Show(ex.Message + " " + ex.StackTrace);
                }
    
            }

            if (invDocs.Count > 0)
            {
                string docs = "";
                foreach (string docname in invDocs)
                {
                    docs = docs + docname + Environment.NewLine;
                }
                MessageBox.Show("Die folgenden Dokumente können nicht verarbeitet werden:\n" + docs, "Info", MessageBoxButtons.OK);
            }

            adCon.CloseDatabase();

           
        }

        private void MoveFilesToUpdated(GutschriftDocumentModel gDoc)
        {
            try
            {
                String[] dateTokens = gDoc.DocumentDate.Split(".".ToCharArray());
                string sortableDate = gDoc.DocumentDate;
                if(dateTokens.Length >= 3)
                {
                    sortableDate = dateTokens[2] + "_" + dateTokens[1] + "_" + dateTokens[0];
                }
                string cc = gDoc.CostCenter;
                if (cc.StartsWith("1"))
                    cc = "ALD";
                Directory.Move(gDoc.PDFFilePath, OCRSettings.settings.OutputDocPath +  sortableDate + "_Gutschrift_" + cc + "_" + gDoc.InvoiceNumber + ".pdf");
            }
            catch(Exception ex)
            {
                MessageBox.Show("Die Datei (" + gDoc.PDFFilePath + "). kann nicht verschoben werden. Eine Datei mit demselben Namen exstiert bereits '" + OCRSettings.settings.OutputDocPath + gDoc.DocumentDate + "_Gutschrift_" + gDoc.InvoiceNumber + ".pdf" + "'\n" + ex.Message + "\n" + ex.StackTrace);
            }
        }

        private void ResetProgressBar()
        {
            prgBarOcr.Value = 0;
            prgBarOcr.Visible = false;
            SetPrgStatus("");
        }

        private void UpdateProgressBar()
        {
            prgBarOcr.Increment(1);
            SetPrgStatus("Processing " + prgBarOcr.Value.ToString() + " of " + prgBarOcr.Maximum.ToString() );
       }

        private void SetPrgStatus(String str1)
        {
            lblPrgStatus.Text = str1;
            lblPrgStatus.Update();

        }

        private void SetStatus(String str1)
        {
            lblStatus.Text = str1;
            lblStatus.Update();
        }

        private void btnStop_Click(object sender, EventArgs e)
        {
            btnStop.Enabled = false;
            btnStart.Enabled = true;
        }

        private void prgBarOcr_Click(object sender, EventArgs e)
        {

        }

        private void OCRForm_Load(object sender, EventArgs e)
        {

        }

        private void grpDoc_Enter(object sender, EventArgs e)
        {

        }

        private void lstDocFiles_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (lstDocFiles.SelectedIndex < 0) return;

            GutschriftDocumentModel gDoc= gutschriftDocList[lstDocFiles.SelectedIndex];

            txtOrder.Text = gDoc.OrderNumber;
            txtInvoice.Text = gDoc.InvoiceNumber;
            txtDate.Text = gDoc.DocumentDate;
            txtCostCenter.Text = gDoc.CostCenter;
            txtSAPNumber.Text = gDoc.SAPNumber;
            txtSumNetto.Text = gDoc.SummeNetto;

            UpdateControlsForValidation(gDoc);

           
            pdfImageBox.OpenPDF(gDoc.PDFFilePath, string.Empty);
            pageCountToolStripLabel.Text = " of " + pdfImageBox.PageCount.ToString();
            pdfImageBox.Update();

            btnUpdate.Enabled = true;

        }

        private void UpdateControlsForValidation(GutschriftDocumentModel gDoc)
        {
            lblDocStatus.Text = "";
            bool hasValidationError = false;

            if (!gDoc.ValidOrderNumber)
            {
                txtOrder.BackColor = Color.FromArgb(255, 80, 80);
                hasValidationError = true;
            }
            else
            {
                txtOrder.BackColor = Color.White;
            }

            if (!gDoc.ValidInvoiceNumber)
            {
                txtInvoice.BackColor = Color.FromArgb(255, 80, 80);
                hasValidationError = true;
            }
            else
            {
                txtInvoice.BackColor = Color.White;
            }

            if (!gDoc.ValidCostCenter)
            {
                txtCostCenter.BackColor = Color.FromArgb(255, 80, 80);
                hasValidationError = true;
            }
            else
            {
                txtCostCenter.BackColor = Color.White;
            }

            if (!gDoc.ValidSAPNumber)
            {
                txtSAPNumber.BackColor = Color.FromArgb(255, 80, 80);
                hasValidationError = true;
            }
            else
            {
                txtSAPNumber.BackColor = Color.White;
            }

            if (!gDoc.ValidDocumentDate)
            {
                txtDate.BackColor = Color.FromArgb(255, 80, 80);
                hasValidationError = true;
            }
            else
            {
                txtDate.BackColor = Color.White;
            }

            if(!gDoc.ValidSummeNetto)
            {
                txtSumNetto.BackColor = Color.FromArgb(255, 80, 80);
                hasValidationError = true;
            }
            else
            {
                txtSumNetto.BackColor = Color.White;
            }

            if (gDoc.pageCount != 1)
            {
                hasValidationError = true;
                lblDocStatus.Text = "Page count error.";
            }

            if (hasValidationError)
            {

                lblDocStatus.Text += " Im Dokument sind problematische Werte vorhanden";
            }
            else
            {
                lblDocStatus.Text = "";
            }
          

        }

        private void btnUpdate_Click(object sender, EventArgs e)
        {
            if (lstDocFiles.SelectedIndex < 0) return;

            GutschriftDocumentModel gDoc = gutschriftDocList[lstDocFiles.SelectedIndex];

            gDoc.OrderNumber = txtOrder.Text; 
            gDoc.InvoiceNumber = txtInvoice.Text ;
            gDoc.DocumentDate=txtDate.Text;
            gDoc.CostCenter = txtCostCenter.Text;
            gDoc.SAPNumber = txtSAPNumber.Text;
            gDoc.SummeNetto= txtSumNetto.Text ;
            gDoc.Validate();

            AccessDbConnector adCon = new AccessDbConnector();
            adCon.ConnectToDatabase();
            gDoc.Validate();
            UpdateControlsForValidation(gDoc);

            if (gDoc.ValidDocument)
            {
                int id = adCon.UpdateInvoiceRecord(gDoc.GetInvoiceData(), gDoc.InitInvNumber);
                if (id > 0)
                {
                    gDoc.ID = id;
                    MoveFilesToUpdated(gDoc);
                }
                gutschriftDocList.Remove(gDoc);

                allDocList.Add(gDoc);
                completedDocCount++;

                pendingDocCount--;
                
                UpdateDocList();

                if (lstDocFiles.Items.Count == 0)
                {
                    MessageBox.Show("All documents updated successfully");
                    
                    this.Close();
                    //ShowCompletedRecords();
                }
                else
                {
                    AutoClosingMessageBox.Show("Document updated successfully","Info",1000);

                    if (lstDocFiles.Items.Count > 0)
                        lstDocFiles.SelectedIndex = 0;
                }
            }
          
            adCon.CloseDatabase();
            
        }

        private void ShowCompletedRecords()
        {
            string idlist = "";
            foreach(GutschriftDocumentModel gDoc in allDocList)
            {
                if (gDoc.ID > 0)
                    idlist += gDoc.ID.ToString() + ",";
            }
            AccessDbConnector adCon = new AccessDbConnector();
            adCon.ConnectToDatabase();
            idlist = idlist.Trim(",".ToCharArray());
            if (!idlist.Trim().Equals(""))
            {
                DataTable tbl = adCon.getInvoiceRecords(idlist);
                frmTableView tview = new frmTableView();
                tview.ShowRecords(tbl);
                tview.Show();
            }
            adCon.CloseDatabase();
        }

        private void showImageRegionToolStripButton_Click(object sender, EventArgs e)
        {
            pdfImageBox.Invalidate();
        }

        private void showSourceImageRegionToolStripButton_Click(object sender, EventArgs e)
        {
            pdfImageBox.Invalidate();
        }

        private void actualSizeToolStripButton_Click(object sender, EventArgs e)
        {
            pdfImageBox.ActualSize();
        }

        private void sizeToFitToolStripButton_Click(object sender, EventArgs e)
        {
            pdfImageBox.SizeToFit = true;
        }

        private void zoomInToolStripButton_Click(object sender, EventArgs e)
        {
            pdfImageBox.ZoomIn();
        }

        private void zoomOutToolStripButton_Click(object sender, EventArgs e)
        {
            pdfImageBox.ZoomOut();
        }

        private void firstPageToolStripButton_Click(object sender, EventArgs e)
        {
            pdfImageBox.FirstPage();
        }

        private void previousPageToolStripButton_Click(object sender, EventArgs e)
        {
            pdfImageBox.PreviousPage();
        }

        private void nextPageToolStripButton_Click(object sender, EventArgs e)
        {
            pdfImageBox.NextPage();
        }

        private void lastPageToolStripButton_Click(object sender, EventArgs e)
        {
            pdfImageBox.LastPage();
        }

        private void pageToolStripTextBox_Click(object sender, EventArgs e)
        {

        }
        private void UpdateUIHandler(object sender, ScrollEventArgs e)
        {
            this.UpdateUI();
        }

        private void pageToolStripTextBox_KeyPress(object sender, KeyPressEventArgs e)
        {
            if (e.KeyChar == '\r')
            {
                int pageNumber;

                e.Handled = true;

                if (int.TryParse(pageToolStripTextBox.Text, out pageNumber) && pageNumber > 0 && pageNumber <= pdfImageBox.PageCount)
                    pdfImageBox.CurrentPage = pageNumber;
                else
                    pageToolStripTextBox.Text = pdfImageBox.CurrentPage.ToString();
            }
        }

        private void UpdateUIHandler(object sender, EventArgs e)
        {
            this.UpdateUI();
        }

        private void pdfImageBox_CurrentPageChanged(object sender, EventArgs e)
        {
            this.UpdateUI();
        }

        private void pdfImageBox_Paint(object sender, PaintEventArgs e)
        {
            // highlight the image
            if (showImageRegionToolStripButton.Checked)
                this.DrawBox(e.Graphics, Color.CornflowerBlue, ((ImageBox)sender).GetImageViewPort());

            // show the region that will be drawn from the source image
            if (showSourceImageRegionToolStripButton.Checked)
                this.DrawBox(e.Graphics, Color.Firebrick, new Rectangle(((ImageBox)sender).GetImageViewPort().Location, ((ImageBox)sender).GetSourceImageRegion().Size));
        }

        private void DrawBox(Graphics graphics, Color color, Rectangle rectangle)
        {
            int offset;
            int penWidth;

            offset = 9;
            penWidth = 2;

            using (SolidBrush brush = new SolidBrush(Color.FromArgb(64, color)))
                graphics.FillRectangle(brush, rectangle);

            using (Pen pen = new Pen(color, penWidth))
            {
                pen.DashStyle = DashStyle.Dot;
                graphics.DrawLine(pen, rectangle.Left, rectangle.Top - offset, rectangle.Left, rectangle.Bottom + offset);
                graphics.DrawLine(pen, rectangle.Left + rectangle.Width, rectangle.Top - offset, rectangle.Left + rectangle.Width, rectangle.Bottom + offset);
                graphics.DrawLine(pen, rectangle.Left - offset, rectangle.Top, rectangle.Right + offset, rectangle.Top);
                graphics.DrawLine(pen, rectangle.Left - offset, rectangle.Bottom, rectangle.Right + offset, rectangle.Bottom);
            }
        }



    }
}
