<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class AuthController extends Controller
{
    /* ============================================================
       USER SIGNUP  (email or phone + OTP)
    ============================================================ */
    public function signup(Request $request)
    {
        $request->validate([
            'name'          => 'required|string|max:255',
            'email_phone'   => 'required',
            'password'      => 'required|min:6|confirmed',
        ]);

        // Detect if email or phone
        $email = null;
        $phone = null;

        if (filter_var($request->email_phone, FILTER_VALIDATE_EMAIL)) {
            $email = $request->email_phone;
        } else {
            $phone = $request->email_phone;
        }

        // Check if email or phone already exists
        if ($email && User::where('email', $email)->exists()) {
            return response()->json(['status' => false, 'message' => 'Email already registered'], 409);
        }

        if ($phone && User::where('phone', $phone)->exists()) {
            return response()->json(['status' => false, 'message' => 'Phone already registered'], 409);
        }

        // Generate 6-digit OTP
        $otp = rand(100000, 999999);

        // Create user
        $user = User::create([
            'name'           => $request->name,
            'email'          => $email,
            'phone'          => $phone,
            'password'       => Hash::make($request->password),
            'otp'            => $otp,
            'otp_expires_at' => Carbon::now()->addMinutes(5),
            'is_verified'    => false,
        ]);

        return response()->json([
            'status'  => true,
            'message' => 'Signup successful. OTP sent.',
            'otp'     => $otp,  // remove in production
            'data'    => $user
        ]);
    }



    /* ============================================================
       VERIFY OTP
    ============================================================ */

    public function verifyOtp(Request $request)
    {
        $request->validate([
            'email_phone' => 'required',
            'otp'         => 'required'
        ]);

        Log::info('verifyOtp called', ['input' => $request->all()]);

        // Detect email or phone
        $email = filter_var($request->email_phone, FILTER_VALIDATE_EMAIL)
                ? $request->email_phone
                : null;

        $phone = !$email ? $request->email_phone : null;

        Log::info('Checking user by email or phone', ['email' => $email, 'phone' => $phone]);

        // Fetch user safely to avoid NULL mismatches
        if ($email) {
            $user = User::where('email', $email)->first();
        } else {
            $user = User::where('phone', $phone)->first();
        }

        if (!$user) {
            Log::warning('User not found', ['email' => $email, 'phone' => $phone]);
            return response()->json(['status' => false, 'message' => 'User not found'], 404);
        }

        Log::info('User found', ['user_id' => $user->id, 'otp' => $user->otp]);

        // Check if OTP matches
        if ($user->otp != $request->otp) {
            Log::warning('Invalid OTP', ['provided' => $request->otp, 'expected' => $user->otp]);
            return response()->json(['status' => false, 'message' => 'Invalid OTP'], 400);
        }

        // Check if OTP expired
        if (now()->greaterThan($user->otp_expires_at)) {
            Log::warning('OTP expired', ['otp_expires_at' => $user->otp_expires_at]);
            return response()->json(['status' => false, 'message' => 'OTP expired'], 400);
        }

        // Verify account
        $user->update([
            'is_verified'    => true,
            'otp'            => null,
            'otp_expires_at' => null
        ]);

        Log::info('OTP verified successfully', ['user_id' => $user->id]);

        return response()->json([
            'status'  => true,
            'message' => 'OTP verified successfully'
        ]);
    }




    /* ============================================================
       USER LOGIN  (email or phone)
    ============================================================ */
    public function login(Request $request)
    {
        $request->validate([
            'login_id' => 'required',
            'password' => 'required'
        ]);

        // Try to find user by email or phone
        $user = User::where('email', $request->login_id)
                    ->orWhere('phone', $request->login_id)
                    ->first();

        if (!$user) {
            return response()->json([
                'status'  => false,
                'message' => 'Invalid email or phone'
            ], 404);
        }

        // Check password
        if (!Hash::check($request->password, $user->password)) {
            return response()->json([
                'status'  => false,
                'message' => 'Wrong password'
            ], 401);
        }

        // Check verification
        if (!$user->is_verified) {
            return response()->json([
                'status'  => false,
                'message' => 'Please verify OTP before logging in'
            ], 401);
        }

        return response()->json([
            'status'  => true,
            'message' => 'Login successful',
            'data'    => $user
        ]);
    }

    /* ============================================================
       FORGOT PASSWORD - REQUEST OTP
    ============================================================ */
    public function forgotPassword(Request $request)
    {
        $request->validate([
            'email_phone' => 'required',
        ]);

        // Detect email or phone
        $email = filter_var($request->email_phone, FILTER_VALIDATE_EMAIL) ? $request->email_phone : null;
        $phone = !$email ? $request->email_phone : null;

        // Find user
        if ($email) {
            $user = User::where('email', $email)->first();
        } else {
            $user = User::where('phone', $phone)->first();
        }

        if (!$user) {
            return response()->json(['status' => false, 'message' => 'User not found'], 404);
        }

        // Generate OTP
        $otp = rand(100000, 999999);

        // Update user with new OTP
        $user->update([
            'otp' => $otp,
            'otp_expires_at' => Carbon::now()->addMinutes(5),
        ]);

        Log::info('Forgot Password OTP generated', ['user_id' => $user->id, 'otp' => $otp]);

        return response()->json([
            'status'  => true,
            'message' => 'OTP sent to your registered contact.',
            'otp'     => $otp, // Remove in production
        ]);
    }

    /* ============================================================
       RESET PASSWORD - VERIFY OTP & UPDATE PASSWORD
    ============================================================ */
    public function resetPassword(Request $request)
    {
        $request->validate([
            'email_phone' => 'required',
            'otp'         => 'required',
            'password'    => 'required|min:6|confirmed', // expects 'password_confirmation' field
        ]);

        // Detect email or phone
        $email = filter_var($request->email_phone, FILTER_VALIDATE_EMAIL) ? $request->email_phone : null;
        $phone = !$email ? $request->email_phone : null;

        // Find user
        if ($email) {
            $user = User::where('email', $email)->first();
        } else {
            $user = User::where('phone', $phone)->first();
        }

        if (!$user) {
            return response()->json(['status' => false, 'message' => 'User not found'], 404);
        }

        // Check OTP
        if ($user->otp != $request->otp) {
            return response()->json(['status' => false, 'message' => 'Invalid OTP'], 400);
        }

        // Check Expiry
        if (now()->greaterThan($user->otp_expires_at)) {
            return response()->json(['status' => false, 'message' => 'OTP expired'], 400);
        }

        // Update Password
        $user->update([
            'password'       => Hash::make($request->password),
            'otp'            => null,
            'otp_expires_at' => null,
        ]);

        Log::info('Password reset successfully', ['user_id' => $user->id]);

        return response()->json([
            'status'  => true,
            'message' => 'Password reset successful. You can now login.',
        ]);
    }
}
