<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\PurchaseStock;
use App\Models\OutwardStock;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class OutwardStockController extends Controller
{
   public function create()
    {
        // Fetch available items with quantities
        $items = PurchaseStock::select(
                'purchase_stock.id as purchase_stock_id',
                'purchase_stock.item_name',
                'purchase_stock.particular_name',
                'purchase_stock.specification',
                'purchase_stock.model_number',
                DB::raw('COALESCE(purchase_stock.qty,0) as purchased_qty'),
                DB::raw('(COALESCE(purchase_stock.qty,0)
                          - COALESCE((SELECT SUM(o.qty) FROM outward_stock o WHERE o.purchase_stock_id = purchase_stock.id),0)
                          - COALESCE((SELECT SUM(f.qty) FROM fault_stock f WHERE f.purchase_stock_id = purchase_stock.id),0)
                         ) as available_qty')
            )
            ->having('available_qty','>',0)
            ->get();

        return view('outward', compact('items'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'outward_date' => 'required|date',
            'given_to' => 'required|string|max:255',
            'purchase_stock_id' => 'required|exists:purchase_stock,id',
            'qty' => 'required|integer|min:1',
            'remarks' => 'nullable|string|max:255',
        ]);

        $purchaseStock = PurchaseStock::findOrFail($request->purchase_stock_id);

        // Check available qty
        $available = $purchaseStock->qty
            - OutwardStock::where('purchase_stock_id',$purchaseStock->id)->sum('qty')
            - DB::table('fault_stock')->where('purchase_stock_id',$purchaseStock->id)->sum('qty');

        if($request->qty > $available){
            return back()->with('error','Requested quantity exceeds available stock!');
        }

        // Generate OS number
        $last = OutwardStock::latest('id')->first();
        $nextNum = $last ? intval(substr($last->os_number,2)) + 1 : 1;
        $os_number = "OS".str_pad($nextNum,6,'0',STR_PAD_LEFT);

        OutwardStock::create([
            'outward_date' => $request->outward_date,
            'given_to' => $request->given_to,
            'purchase_stock_id' => $purchaseStock->id,
            'item_name' => $purchaseStock->item_name,
            'particular_name' => $purchaseStock->particular_name,
            'specification' => $purchaseStock->specification,
            'model_number' => $purchaseStock->model_number,
            'qty' => $request->qty,
            'remarks' => $request->remarks,
            'os_number' => $os_number,
        ]);

        return redirect()->route('outward_view')->with('success','Outward Stock Added Successfully!');
    }

     public function index(Request $request)
    {
        $query = OutwardStock::with('purchaseStock');

        // Filters
        if ($request->filled('item_name')) {
            $query->whereHas('purchaseStock', function($q) use ($request) {
                $q->where('item_name', 'like', '%'.$request->item_name.'%');
            });
        }
        if ($request->filled('given_to')) {
            $query->where('given_to', 'like', '%'.$request->given_to.'%');
        }
        if ($request->filled('outward_date')) {
            $query->whereDate('outward_date', $request->outward_date);
        }

        $stocks = $query->orderByDesc('id')->paginate(10);

        return view('outward_view', compact('stocks'));
    }

    public function destroy($id)
    {
        $stock = OutwardStock::findOrFail($id);
        $stock->delete();

        return response()->json(['success' => true]);
    }
public function edit($id)
{
    $stock = OutwardStock::with('purchaseStock')->findOrFail($id);

    Log::info("Stock info".$stock);

    $purchaseStock = $stock->purchaseStock;

 Log::info("purchaseStock info".  $purchaseStock );
    $purchasedQty = $purchaseStock->qty ?? 0;

     Log::info("purchaseqty info".   $purchasedQty);

    $outwardSum = OutwardStock::where('purchase_stock_id', $purchaseStock->id)
                    ->sum('qty');

    Log::info("outwardsum info".   $outwardSum);

    $faultSum = DB::table('fault_stock')->where('purchase_stock_id', $purchaseStock->id)->sum('qty');

  Log::info("faultSum info".   $faultSum);

    $available_qty = $purchasedQty - $outwardSum - $faultSum ;
    
Log::info("available_qty info".    $available_qty );


    return view('outward_edit', compact('stock', 'available_qty'));
}

public function update(Request $request, $id)
{
    $request->validate([
        'outward_date' => 'required|date',
        'given_to' => 'required|string|max:255',
        'item_name' => 'required|string|max:255',
        'particular_name' => 'nullable|string|max:255',
        'specification' => 'nullable|string|max:255',
        'model_number' => 'nullable|string|max:255',
        'qty' => 'required|integer|min:1',
        'remarks' => 'nullable|string|max:255',
    ]);

    $stock = OutwardStock::findOrFail($id);
    $purchaseStock = $stock->purchaseStock;

    // Available qty check
    $purchasedQty = $purchaseStock->qty ?? 0;
    $outwardSum = OutwardStock::where('purchase_stock_id', $purchaseStock->id)
                    ->where('id', '!=', $stock->id)
                    ->sum('qty');
    $faultSum = DB::table('fault_stock')->where('purchase_stock_id', $purchaseStock->id)->sum('qty');
    $available_qty = $purchasedQty - $outwardSum - $faultSum + $stock->qty;

    if ($request->qty > $available_qty) {
        return back()->with('error', "Quantity exceeds available stock ($available_qty).");
    }

    $stock->update([
        'outward_date' => $request->outward_date,
        'given_to' => $request->given_to,
        'item_name' => $request->item_name,
        'particular_name' => $request->particular_name,
        'specification' => $request->specification,
        'model_number' => $request->model_number,
        'qty' => $request->qty,
        'remarks' => $request->remarks,
    ]);

    return redirect()->route('outward_view')->with('success', 'Outward Stock Updated Successfully!');
}


}
